/* PSIPRED3 - Neural Network Prediction of Secondary Structure */

/* Copyright (C) 2000 David T. Jones - Created : January 2000 */
/* Original Neural Network code Copyright (C) 1990 David T. Jones */

/* 2nd Level Prediction Module */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <time.h>

#include "ssdefs.h"
#include "sspred_net2.h"
//#define DEBUG

#define MAXSEQLEN 10000
int numWeightFile = 0;

char           *wtfnm;

int             nwtsum, fwt_to[TOTAL], lwt_to[TOTAL];
float            activation[TOTAL], bias[TOTAL], *weight[TOTAL];

float           profile[MAXSEQLEN][3];

char            seq[MAXSEQLEN];

int             seqlen, nprof;


void err(char *s)
{
    fprintf(stderr, "%s\n", s);
}

void fail(char *s)
{
    err(s);
    exit(1);
}

char *rootname(const char* filename, char* rtname)/*{{{*/
/*****************************************************************************
 * rootname
 * given the file name, 
 * return the rootname of the filename
 ****************************************************************************/
{
    const char *pch;
    char *pstr;
    if((pch = strrchr(filename,'/')) != NULL) {
        pstr = (char*) pch+1;
    } else { 
        pstr = (char*) filename;
    }

    if((pch = strrchr(pstr,'.')) != NULL) {
        strncpy(rtname,pstr, (int)(pch - pstr));
        rtname[pch-pstr]='\0';
    } else {
        rtname = pstr;
    }
    return rtname;
}
/*}}}*/
char** ReadFileList(const char* filename, char **filenameList, int* p_cntfile)/*{{{*/
{
    int cntfile=0;
    int i = 0;
    FILE* fpin = fopen(filename,"r");
    if( fpin == NULL) {
        fail("Unable to open listFile!");
    }
    if (fseek(fpin, 0 , SEEK_END) != 0) {
        fail("fseek of the listFile failed");
    }
    int filesize = ftell(fpin);
    char *string = 0;
    /*read in the while file to string*/
    string = malloc((sizeof(char)*(filesize+1)));
    if (fseek(fpin,0,SEEK_SET) != 0) {
        fail("fseek of the listFile failed");
    }
    if (fread(string, sizeof(char),filesize, fpin) != filesize) {
        fail("fread of the listFile failed");
    }
    fclose(fpin);
    string[filesize]=0;


    /*get the number of lines*/
    int cntline=0;
    for (i = 0; i < filesize; i++) {
        if (string[i] == '\n') {
            cntline++;
        }
    }
    filenameList=malloc(sizeof(char*)*(cntline+1));
    /*parse the string to filenameList*/
    char *pch;
    pch = strtok(string, "\n");
    int n=0;
    while(pch!=NULL)
    {
        n = strlen(pch);
        if (n > 0)
        {
            filenameList[cntfile] = malloc(sizeof(char)*(n+1));
            strcpy(filenameList[cntfile], pch);
            cntfile ++;
        }
        pch = strtok(NULL,"\n");
    }
#ifdef DEBUG
    for (i=0;i<cntfile; i++)
    {
        fprintf(stderr,"%d:%s\n", i, filenameList[i]);
    }
#endif 
    free(string);
    *p_cntfile = cntfile;
    return filenameList;

}/*}}}*/

void compute_output(void)
{
    int             i, j;
    float            netinp, sum, omax;

    for (i = NUM_IN; i < NUM_IN + NUM_HID; i++)
    {
        netinp = bias[i];

        for (j = fwt_to[i]; j < lwt_to[i]; j++)
            netinp += activation[j] * weight[i][j];

        /* Trigger neuron */
        activation[i] = logistic(netinp);
    }

    for (i = NUM_IN + NUM_HID; i < TOTAL; i++)
    {
        netinp = bias[i];

        for (j = fwt_to[i]; j < lwt_to[i]; j++)
            netinp += activation[j] * weight[i][j];

        /* Trigger neuron */
        activation[i] = logistic(netinp);
    }
}

/*
 * load weights - load all link weights from a disk file
 */
void load_wts(char *fname)
{
    int             i, j;
    double          t, chksum = 0.0;
    FILE           *ifp;

    if (!(ifp = fopen(fname, "r")))
	fail("Cannot open weight file!\n");

    /* Load input units to hidden layer weights */
    for (i = NUM_IN; i < NUM_IN + NUM_HID; i++)
	for (j = fwt_to[i]; j < lwt_to[i]; j++)
	{
	    fscanf(ifp, "%lf", &t);
	    weight[i][j] = t;
	    chksum += t*t;
	}

    /* Load hidden layer to output units weights */
    for (i = NUM_IN + NUM_HID; i < TOTAL; i++)
	for (j = fwt_to[i]; j < lwt_to[i]; j++)
	{
	    fscanf(ifp, "%lf", &t);
	    weight[i][j] = t;
	    chksum += t*t;
	}

    /* Load bias weights */
    for (j = NUM_IN; j < TOTAL; j++)
    {
	fscanf(ifp, "%lf", &t);
	bias[j] = t;
	chksum += t*t;
    }

    /* Read expected checksum at end of file */
    if (fscanf(ifp, "%lf", &t) != 1 || ferror(ifp))
	fail("Weight file read error!");

    fclose(ifp);

    if ((int)t != (int)(chksum+0.5))
	fail("Corrupted weight file detected!");
}

void init(void)
{
    int             i, j;

    for (i = NUM_IN; i < TOTAL; i++) {
        if (!(weight[i] = calloc(TOTAL - NUM_OUT, sizeof(float)))){
            fail("init: Out of Memory!");
        }
    }

    /* Connect input units to hidden layer */
    for (i = NUM_IN; i < NUM_IN + NUM_HID; i++)
    {
        fwt_to[i] = 0;
        lwt_to[i] = NUM_IN;
    }

    /* Connect hidden units to output layer */
    for (i = NUM_IN + NUM_HID; i < TOTAL; i++)
    {
        fwt_to[i] = NUM_IN;
        lwt_to[i] = NUM_IN + NUM_HID;
    }
}

/* Convert AA letter to numeric code (0-20) */
int aanum(int ch)
{
    static int      aacvs[] =
    {
	999, 0, 20, 4, 3, 6, 13, 7, 8, 9, 20, 11, 10, 12, 2,
	20, 14, 5, 1, 15, 16, 20, 19, 17, 20, 18, 20
    };

    return (isalpha(ch) ? aacvs[ch & 31] : 20);
}

void predict_new(int niters, float dca, float dcb, char *outname, FILE*fpHoriz)/*{{{*/
{
    int             aa, a, b, nb, i, j, k, n, winpos;
    char            pred, predsst[MAXSEQLEN], lastpreds[MAXSEQLEN], *che = "CHE";
    float           score_c[MAXSEQLEN], score_h[MAXSEQLEN], score_e[MAXSEQLEN], bestsc, score, conf[MAXSEQLEN], predq3, av_c, av_h, av_e;
    FILE *ofp;
    strcpy(predsst,"");
    strcpy(lastpreds,"");

    ofp = fopen(outname, "w");
    if (!ofp)
      fail("Cannot open output file!");

    fputs("# PSIPRED VFORMAT (PSIPRED V3.2)\n\n", ofp);
    
    if (niters < 1)
      niters = 1;

    do {
	memcpy(lastpreds, predsst, seqlen);
	av_c = av_h = av_e = 0.0;
	for (winpos = 0; winpos < seqlen; winpos++)
	{
	    av_c += profile[winpos][0];
	    av_h += profile[winpos][1];
	    av_e += profile[winpos][2];
	}
	av_c /= seqlen;
	av_h /= seqlen;
	av_e /= seqlen;
	for (winpos = 0; winpos < seqlen; winpos++)
    {
        for (j = 0; j < NUM_IN; j++)
            activation[j] = 0.0;
        activation[(WINR - WINL + 1) * IPERGRP] = av_c;
        activation[(WINR - WINL + 1) * IPERGRP + 1] = av_h;
        activation[(WINR - WINL + 1) * IPERGRP + 2] = av_e;
        activation[(WINR - WINL + 1) * IPERGRP + 3] = log((double)seqlen);
        for (j = WINL; j <= WINR; j++)
        {
            if (j + winpos >= 0 && j + winpos < seqlen)
            {
                for (aa = 0; aa < 3; aa++)
                    activation[(j - WINL) * IPERGRP + aa] = profile[j + winpos][aa];
            }
            else
                activation[(j - WINL) * IPERGRP + 3] = 1.0;
        }
        compute_output();
        if (activation[TOTAL - NUM_OUT] > dca * activation[TOTAL - NUM_OUT + 1] && activation[TOTAL - NUM_OUT] > dcb * activation[TOTAL - NUM_OUT + 2]){
            pred = 'C';
        }
        else if (dca * activation[TOTAL - NUM_OUT + 1] > activation[TOTAL - NUM_OUT] && dca * activation[TOTAL - NUM_OUT + 1] > dcb * activation[TOTAL - NUM_OUT + 2]){
            pred = 'H';
        }
        else{
            pred = 'E';
        }
        predsst[winpos] = pred;
        score_c[winpos] = activation[TOTAL - NUM_OUT];
        score_h[winpos] = activation[TOTAL - NUM_OUT + 1];
        score_e[winpos] = activation[TOTAL - NUM_OUT + 2];
    }
	
	for (winpos = 0; winpos < seqlen; winpos++)
	{
	    profile[winpos][0] = score_c[winpos];
	    profile[winpos][1] = score_h[winpos];
	    profile[winpos][2] = score_e[winpos];
	}
    } while (memcmp(predsst, lastpreds, seqlen) && --niters);
    
    for (winpos = 0; winpos < seqlen; winpos++)
	conf[winpos] = (2*MAX(MAX(score_c[winpos], score_h[winpos]), score_e[winpos])-(score_c[winpos]+score_h[winpos]+score_e[winpos])+MIN(MIN(score_c[winpos], score_h[winpos]), score_e[winpos]));
    
    /* Filter remaining singleton assignments */
    for (winpos = 0; winpos < seqlen; winpos++)
	if (winpos && winpos < seqlen - 1 && predsst[winpos - 1] == predsst[winpos + 1] && conf[winpos] < 0.5*(conf[winpos-1]+conf[winpos+1])){
        predsst[winpos] = predsst[winpos - 1];
    }
	    
    
    for (winpos = 0; winpos < seqlen; winpos++)
    {
        if (winpos && winpos < seqlen - 1 && predsst[winpos - 1] == 'C' && predsst[winpos] != predsst[winpos + 1]){
            predsst[winpos] = 'C';
        }

        if (winpos && winpos < seqlen - 1 && predsst[winpos + 1] == 'C' && predsst[winpos] != predsst[winpos - 1]){
            predsst[winpos] = 'C';
        }
    }
    
    for (winpos=0; winpos<seqlen; winpos++)
	fprintf(ofp, "%4d %c %c  %6.3f %6.3f %6.3f\n", winpos + 1, seq[winpos], predsst[winpos], score_c[winpos], score_h[winpos], score_e[winpos]);
    
    fclose(ofp);
    
    nb = seqlen / 60 + 1;
    j = 1;
    for (b = 0; b < nb; b++)
    {
	fprintf(fpHoriz, "\nConf: ");
	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    //putchar(MIN((char)(10.0*conf[j-1]+'0'), '9'));
	    fprintf(fpHoriz,"%c", MIN((char)(10.0*conf[j-1]+'0'), '9'));
	}

	fprintf(fpHoriz, "\nPred: ");

	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    //putchar(predsst[j - 1]);
		fprintf(fpHoriz, "%c", predsst[j - 1]);
	}

	fprintf(fpHoriz, "\n  AA: ");

	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    //putchar(seq[j - 1]);
	    fprintf(fpHoriz, "%c", seq[j - 1]);
	}

	fprintf(fpHoriz, "\n      ");

	for (i = 0; i < 58; i++)
	{
	    if (b * 60 + i + 3 > seqlen)
		break;
	    j = b * 60 + i + 3;
	    if (!(j % 10))
	    {
		fprintf(fpHoriz,"%3d", j);
		i += 2;
	    }
	    else
		fprintf(fpHoriz, " ");
	}
	fprintf(fpHoriz, "\n");

	fprintf(fpHoriz, "\n");
    }
}/*}}}*/
/* Main prediction routine */
void predict(int niters, float dca, float dcb, char *outname)/*{{{*/
{
    int             aa, a, b, nb, i, j, k, n, winpos;
    char            pred, predsst[MAXSEQLEN], lastpreds[MAXSEQLEN], *che = "CHE";
    float           score_c[MAXSEQLEN], score_h[MAXSEQLEN], score_e[MAXSEQLEN], bestsc, score, conf[MAXSEQLEN], predq3, av_c, av_h, av_e;
    FILE *ofp;
    strcpy(predsst,"");
    strcpy(lastpreds,"");

    ofp = fopen(outname, "w");
    if (!ofp)
      fail("Cannot open output file!");

    fputs("# PSIPRED VFORMAT (PSIPRED V3.2)\n\n", ofp);
    
    if (niters < 1)
      niters = 1;

    do {
	memcpy(lastpreds, predsst, seqlen);
	av_c = av_h = av_e = 0.0;
	for (winpos = 0; winpos < seqlen; winpos++)
	{
	    av_c += profile[winpos][0];
	    av_h += profile[winpos][1];
	    av_e += profile[winpos][2];
	}
	av_c /= seqlen;
	av_h /= seqlen;
	av_e /= seqlen;
	for (winpos = 0; winpos < seqlen; winpos++)
	{
	    for (j = 0; j < NUM_IN; j++)
		activation[j] = 0.0;
	    activation[(WINR - WINL + 1) * IPERGRP] = av_c;
	    activation[(WINR - WINL + 1) * IPERGRP + 1] = av_h;
	    activation[(WINR - WINL + 1) * IPERGRP + 2] = av_e;
	    activation[(WINR - WINL + 1) * IPERGRP + 3] = log((double)seqlen);
	    for (j = WINL; j <= WINR; j++)
	    {
		if (j + winpos >= 0 && j + winpos < seqlen)
		{
		    for (aa = 0; aa < 3; aa++)
			activation[(j - WINL) * IPERGRP + aa] = profile[j + winpos][aa];
		}
		else
		    activation[(j - WINL) * IPERGRP + 3] = 1.0;
	    }
	    compute_output();
	    if (activation[TOTAL - NUM_OUT] > dca * activation[TOTAL - NUM_OUT + 1] && activation[TOTAL - NUM_OUT] > dcb * activation[TOTAL - NUM_OUT + 2])
		pred = 'C';
	    else if (dca * activation[TOTAL - NUM_OUT + 1] > activation[TOTAL - NUM_OUT] && dca * activation[TOTAL - NUM_OUT + 1] > dcb * activation[TOTAL - NUM_OUT + 2])
		pred = 'H';
	    else
		pred = 'E';
	    predsst[winpos] = pred;
	    score_c[winpos] = activation[TOTAL - NUM_OUT];
	    score_h[winpos] = activation[TOTAL - NUM_OUT + 1];
	    score_e[winpos] = activation[TOTAL - NUM_OUT + 2];
	}
	
	for (winpos = 0; winpos < seqlen; winpos++)
	{
	    profile[winpos][0] = score_c[winpos];
	    profile[winpos][1] = score_h[winpos];
	    profile[winpos][2] = score_e[winpos];
	}
    } while (memcmp(predsst, lastpreds, seqlen) && --niters);
    
    for (winpos = 0; winpos < seqlen; winpos++)
	conf[winpos] = (2*MAX(MAX(score_c[winpos], score_h[winpos]), score_e[winpos])-(score_c[winpos]+score_h[winpos]+score_e[winpos])+MIN(MIN(score_c[winpos], score_h[winpos]), score_e[winpos]));
    
    /* Filter remaining singleton assignments */
    for (winpos = 0; winpos < seqlen; winpos++)
	if (winpos && winpos < seqlen - 1 && predsst[winpos - 1] == predsst[winpos + 1] && conf[winpos] < 0.5*(conf[winpos-1]+conf[winpos+1]))
	    predsst[winpos] = predsst[winpos - 1];
    
    for (winpos = 0; winpos < seqlen; winpos++)
    {
	if (winpos && winpos < seqlen - 1 && predsst[winpos - 1] == 'C' && predsst[winpos] != predsst[winpos + 1])
	    predsst[winpos] = 'C';
	if (winpos && winpos < seqlen - 1 && predsst[winpos + 1] == 'C' && predsst[winpos] != predsst[winpos - 1])
	    predsst[winpos] = 'C';
    }
    
    for (winpos=0; winpos<seqlen; winpos++)
	fprintf(ofp, "%4d %c %c  %6.3f %6.3f %6.3f\n", winpos + 1, seq[winpos], predsst[winpos], score_c[winpos], score_h[winpos], score_e[winpos]);
    
    fclose(ofp);
    
    nb = seqlen / 60 + 1;
    j = 1;
    for (b = 0; b < nb; b++)
    {
	printf("\nConf: ");
	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    putchar(MIN((char)(10.0*conf[j-1]+'0'), '9'));
	}

	printf("\nPred: ");

	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    putchar(predsst[j - 1]);
	}

	printf("\n  AA: ");

	for (i = 0; i < 60; i++)
	{
	    if (b * 60 + i >= seqlen)
		break;
	    j = b * 60 + i + 1;
	    putchar(seq[j - 1]);
	}

	printf("\n      ");

	for (i = 0; i < 58; i++)
	{
	    if (b * 60 + i + 3 > seqlen)
		break;
	    j = b * 60 + i + 3;
	    if (!(j % 10))
	    {
		printf("%3d", j);
		i += 2;
	    }
	    else
		printf(" ");
	}
	putchar('\n');

	putchar('\n');
    }
}
/*}}}*/
/* Read PSI AA frequency data */
int getss(FILE * lfil)/*{{{*/
{
    int             i, j, naa;
    float pv[3];
    char            buf[256], *p;
#ifdef DEBUG
    fprintf(stderr,"nprof=%d\n", nprof);
#endif

    naa = 0;
    while (!feof(lfil))
    {
	if (!fgets(buf, 256, lfil))
	    break;
	seq[naa] = buf[5];
	if (sscanf(buf + 11, "%f%f%f", &pv[0], &pv[1], &pv[2]) != 3)
	    break;
	
	if (!nprof)
	{
	    profile[naa][0] = pv[0];
	    profile[naa][1] = pv[1];
	    profile[naa][2] = pv[2];
	}
	else
	{
	    profile[naa][0] += pv[0];
	    profile[naa][1] += pv[1];
	    profile[naa][2] += pv[2];
	}
	
	naa++;
    }
    
    nprof++;
    
    if (!naa)
	fail("Bad psipred pass1 file format!");
    
    return naa;
}/*}}}*/
void PrintHelp()
{
    printf("usage: psipass2 [options] weight-file itercount DCA DCB [outputfile ss-infile or -l listfile ]\n");
    printf("\n");
    printf("options:\n");
    printf("    -outpath <dir> : set output path, default=./\n");
    printf("    -l <file>      : set the list file\n");
    printf("    -h|--help  : print this help message\n");
    printf("\n");
    printf("updated 2010-12-03, Nanjiang\n");
}

int main(int argc, char **argv)
{
    int             i;
    FILE           *ifp;

    /* malloc_debug(3); */
    if (argc < 7)
	{
		PrintHelp();
		exit(1);
	}

	int cnt = 0;/**/
	char listFile[500]="";
	char outpath[500]="./";
	char ssFile[500] = "";
	char outFile[500] = "";
	char weightFile[500] = "";
	int niters=0;
	float dca=0;
	float dcb=0;

    i = 1;
	while(i < argc )
	{
		if (argv[i][0] == '-')
		{
			if(strcmp(argv[i],"-h") == 0 ||strcmp(argv[i],"--help")==0 )
			{
				PrintHelp();
				break;
			}
			else if(strcmp(argv[i],"-outpath") == 0||strcmp(argv[i],"--outpath") == 0)
			{
				strcpy(outpath,argv[i+1]);
				i += 2;
			}
			else if(strcmp(argv[i],"-l") == 0||strcmp(argv[i],"--l") == 0)
			{
				strcpy(listFile,argv[i+1]);
				i += 2;
			}
			else
			{
				fprintf(stderr,"Wrong argument:%s\n",argv[i]);
				exit(1);
			}
		}
	    else
        {
			if (cnt==0) {
				strcpy(weightFile, argv[i]);
				cnt++;
			}
			else if (cnt==1) {
				niters = atoi(argv[i]);
				cnt++;
			}
			else if (cnt==2) {
				dca = atoi(argv[i]);
				cnt++;
			}
			else if (cnt==3) {
				dcb = atoi(argv[i]);
				cnt++;
			}
			else if (cnt==4) {
				strcpy(outFile, argv[i]);
				cnt++;
			}
			else if (cnt==5) {
				strcpy(ssFile, argv[i]);
				cnt++;
			}
			else if (cnt>5)
			{
				fprintf(stderr,"Too many argument! %s", argv[i]);
				exit(1);
			}
            i += 1;
        }
	}
	numWeightFile=cnt ;
	/*fail("usage : psipass2 weight-file itercount DCA DCB outputfile ss-infile ...");*/
	for (i=0;i< TOTAL;i++) {
		weight[i]=NULL;
	}

    init();
    /*load_wts(wtfnm = argv[1]);*/
    load_wts(weightFile);
	if (strcmp(ssFile, "") != 0)/*{{{*/
	{
		ifp = fopen(ssFile, "r");
		if (!ifp)
			fail("Cannot open input file!");
#ifdef DEBUG
        fprintf(stderr,"nprof=%d\n", nprof);
#endif
		seqlen = getss(ifp);
		fclose(ifp);

		for (i=0; i<seqlen; i++)
		{
			profile[i][0] /= nprof;
			profile[i][1] /= nprof;
			profile[i][2] /= nprof;
		}

		puts("# PSIPRED HFORMAT (PSIPRED V3.2)");
		/*predict(atoi(argv[2]), (float)atof(argv[3]), (float)atof(argv[4]), argv[5]);*/
		predict(niters, dca, dcb, outFile);
	}/*}}}*/
	else if (strcmp(listFile,"") != 0)/*{{{*/
	{
        char cmd[500]="";
        sprintf(cmd, "mkdir -p %s", outpath);
        system(cmd);

        int cntfile=0;
        char **filenameList=NULL;
        filenameList = ReadFileList(listFile,filenameList, &cntfile);
        char rtname[500]="";
        char ss2File[500]="";
        char horizFile[500]="";
        /*fprintf(stderr,"cntfile=%d\n", cntfile);*/
        int ifile;
        for (ifile = 0; ifile < cntfile ; ifile ++)
		{
            nprof = 0;
            seqlen = 0;
			ifp = fopen(filenameList[ifile], "r");
			if (!ifp)  {
                fail("Cannot open input file!");
            }
			seqlen = getss(ifp);
			fclose(ifp);


            rootname(filenameList[ifile], rtname);
            sprintf(ss2File,"%s/%s.ss2", outpath, rtname);
            sprintf(horizFile,"%s/%s.horiz", outpath, rtname);
			FILE *fpHoriz = fopen(horizFile, "w");

			for (i=0; i<seqlen; i++)
			{
				profile[i][0] /= nprof;
				profile[i][1] /= nprof;
				profile[i][2] /= nprof;
			}
#ifdef DEBUG
            fprintf(stderr,"file : %s, seqlen =%d\n", filenameList[ifile], seqlen);
            for (i=0;i<seqlen;i++){
                fprintf(stderr,"%4d %c %.3f %.3f %.3f\n", i+1, seq[i], profile[i][0],profile[i][1],profile[i][2]);
            }
#endif
			fprintf(fpHoriz,"# PSIPRED HFORMAT (PSIPRED V3.2)\n");
			/*predict(atoi(argv[2]), (float)atof(argv[3]), (float)atof(argv[4]), argv[5]);*/
			predict_new(niters, dca, dcb, ss2File, fpHoriz);
			fclose(fpHoriz);
		}
        for (ifile = 0; ifile<cntfile;ifile++)
        {
            free(filenameList[ifile]);
        }
        free(filenameList);
	}/*}}}*/
    
	for (i=0;i<TOTAL;i++) {
		if(weight[i]!= NULL) {
			free(weight[i]);
		}
	}
    return 0;
}
